<?php

namespace Unific\Connector\Connection;

use Laminas\Http\HeadersFactory;
use Laminas\Http\RequestFactory;
use Laminas\Http\Response;
use Laminas\Http\Client;
use Laminas\Http\Client\Adapter\Curl;
use Laminas\Http\Request;
use Laminas\Stdlib\ParametersFactory;

class Connection implements ConnectionInterface
{
    protected $urlData = [];

    protected $httpClient;
    protected $httpRequestFactory;
    protected $httpHeadersFactory;
    protected $parametersFactory;

    /**
     * Connection constructor.
     *
     * @param Client $httpClient
     * @param RequestFactory $httpRequestFactory
     * @param HeadersFactory $httpHeadersFactory
     */
    public function __construct(
        Client $httpClient,
        RequestFactory $httpRequestFactory,
        HeadersFactory $httpHeadersFactory,
        ParametersFactory $parametersFactory
    ) {
        $this->httpClient = $httpClient;
        $this->httpRequestFactory = $httpRequestFactory;
        $this->httpHeadersFactory = $httpHeadersFactory;
        $this->parametersFactory = $parametersFactory;
    }

    public function setup()
    {
        return $this;
    }

    public function doRequest()
    {
        return $this;
    }

    public function handleResponse()
    {
        return $this;
    }

    /**
     * @return mixed|string
     */
    public function getRestPath()
    {
        return (isset($this->urlData['query']))
            ? $this->urlData['path'] . '?' . urlencode($this->urlData['query']) : $this->urlData['path'];
    }

    /**
     * @param $url
     * @param array $data
     * @param array $extraHeaders
     * @param $requestType
     * @return Response
     */
    public function sendData($url, $data = [], $extraHeaders = [], $requestType = Request::METHOD_POST)
    {
        $this->httpClient->setOptions(['sslverifypeer' => null, 'sslallowselfsigned' => true]);
        $this->httpClient->setAdapter(new Curl());
        $this->httpClient->setEncType(Client::ENC_FORMDATA);
        $this->httpClient->setUri($url);

        $httpHeaders = $this->httpHeadersFactory->create();
        $httpHeaders->addHeaders($extraHeaders);

        $httpRequest = $this->httpRequestFactory->create();
        $httpRequest->setUri($url);
        $httpRequest->setMethod($requestType);
        $httpRequest->setHeaders($httpHeaders);

        switch ($requestType) {
            case Request::METHOD_POST:
            case Request::METHOD_PUT:
                $httpRequest->setContent(json_encode($data));
                break;
            default:
                $query = $this->parametersFactory->create(['values' => $data]);
                $httpRequest->setQuery($query);
                break;
        }

        return $this->httpClient->send($httpRequest);
    }

    /**
     * @param $url
     * @param array $data
     * @param array $headers
     * @return mixed
     */
    public function post($url, $data = [], $headers = [])
    {
        return $this->sendData($url, $data, $headers, Request::METHOD_POST);
    }

    /**
     * @param $url
     * @param array $data
     * @param array $headers
     * @return mixed
     */
    public function get($url, $data = [], $headers = [])
    {
        return $this->sendData($url, $data, $headers, Request::METHOD_GET);
    }

    /**
     * @param $url
     * @param array $data
     * @param array $headers
     * @return mixed
     */
    public function put($url, $data = [], $headers = [])
    {
        return $this->sendData($url, $data, $headers, Request::METHOD_PUT);
    }

    /**
     * @param $url
     * @param array $data
     * @param array $headers
     * @return mixed
     */
    public function delete($url, $data = [], $headers = [])
    {
        return $this->sendData($url, $data, $headers, Request::METHOD_DELETE);
    }
}
