<?php

namespace Unific\Connector\Helper;

use Laminas\Http\Request;
use Magento\Catalog\Model\ResourceModel\Category\Collection as CategoryCollection;
use Magento\Catalog\Model\ResourceModel\Category\CollectionFactory as CategoryCollectionFactory;
use Magento\Catalog\Model\ResourceModel\Product\Collection as ProductCollection;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory as ProductCollectionFactory;
use Magento\Customer\Model\ResourceModel\CustomerRepository;
use Magento\Customer\Model\ResourceModel\Customer\CollectionFactory as CustomerCollectionFactory;
use Magento\Sales\Model\OrderRepository;
use Magento\Sales\Model\ResourceModel\Order\CollectionFactory as OrderCollectionFactory;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Framework\App\Helper\Context;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Store\Api\Data\StoreInterface;
use Magento\Store\Model\App\Emulation;
use Magento\Store\Model\StoreManagerInterface;
use Unific\Connector\Helper\Data\Cart;
use Unific\Connector\Helper\Data\Category;
use Unific\Connector\Helper\Data\Customer;
use Unific\Connector\Helper\Data\Order;
use Unific\Connector\Helper\Data\Product;
use Unific\Connector\Helper\Message\Queue;
use Unific\Connector\Model\HistoricalFactory;
use Unific\Connector\Model\ResourceModel\Historical\Collection as HistoricalCollection;
use Unific\Connector\Model\ResourceModel\Historical\CollectionFactory as HistoricalCollectionFactory;
use Unific\Connector\Model\ResourceModel\Queue\Collection as QueueCollection;
use Unific\Connector\Model\ResourceModel\Queue\CollectionFactory as QueueCollectionFactory;

class Historical extends AbstractHelper
{
    /**
     * @var Message\Queue
     */
    protected $queueHelper;
    /**
     * @var QueueCollectionFactory
     */
    protected $queueCollectionFactory;
    /**
     * @var HistoricalCollectionFactory
     */
    protected $historicalCollectionFactory;
    /**
     * @var HistoricalFactory
     */
    protected $historicalFactory;
    /**
     * @var SearchCriteriaBuilder
     */
    protected $searchCriteriaBuilder;
    /**
     * @var OrderRepository
     */
    protected $orderRepository;
    /**
     * @var CustomerRepository
     */
    protected $customerRepository;
    /**
     * @var CategoryCollectionFactory
     */
    protected $categoryCollectionFactory;
    /**
     * @var ProductCollectionFactory
     */
    protected $productCollectionFactory;
    /**
     * @var Data\Order
     */
    protected $orderDataHelper;
    /**
     * @var Data\Customer
     */
    protected $customerDataHelper;
    /**
     * @var Data\Cart
     */
    protected $cartDataHelper;
    /**
     * @var Data\Product
     */
    protected $productDataHelper;
    /**
     * @var Data\Category
     */
    protected $categoryDataHelper;
    /**
     * @var Emulation
     */
    protected $emulation;
    /**
     * @var StoreManagerInterface
     */
    protected $storeManager;
    /**
     * @var CustomerCollectionFactory
     */
    protected $customerCollectionFactory;
    /**
     * @var OrderCollectionFactory
     */
    protected $orderCollectionFactory;

    /**
     * @param Context $context
     * @param QueueCollectionFactory $queueCollectionFactory
     * @param HistoricalCollectionFactory $historicalCollectionFactory
     * @param HistoricalFactory $historicalFactory
     * @param Queue $queueHelper
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param OrderRepository $orderRepository
     * @param CustomerRepository $customerRepository
     * @param ProductCollectionFactory $productCollectionFactory
     * @param CategoryCollectionFactory $categoryCollectionFactory
     * @param Order $orderDataHelper
     * @param Cart $cartDataHelper
     * @param Customer $customerDataHelper
     * @param Product $productDataHelper
     * @param Category $categoryDataHelper
     * @param Emulation $emulation
     * @param StoreManagerInterface $storeManager
     * @param CustomerCollectionFactory $customerCollectionFactory
     * @param OrderCollectionFactory $orderCollectionFactory
     */
    public function __construct(
        Context                     $context,
        QueueCollectionFactory      $queueCollectionFactory,
        HistoricalCollectionFactory $historicalCollectionFactory,
        HistoricalFactory           $historicalFactory,
        Queue                       $queueHelper,
        SearchCriteriaBuilder       $searchCriteriaBuilder,
        OrderRepository             $orderRepository,
        CustomerRepository          $customerRepository,
        ProductCollectionFactory    $productCollectionFactory,
        CategoryCollectionFactory   $categoryCollectionFactory,
        Order                       $orderDataHelper,
        Cart                        $cartDataHelper,
        Customer                    $customerDataHelper,
        Product                     $productDataHelper,
        Category                    $categoryDataHelper,
        Emulation                   $emulation,
        StoreManagerInterface       $storeManager,
        CustomerCollectionFactory   $customerCollectionFactory,
        OrderCollectionFactory      $orderCollectionFactory
    ) {
        parent::__construct($context);

        $this->queueHelper = $queueHelper;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->orderRepository = $orderRepository;
        $this->customerRepository = $customerRepository;
        $this->productCollectionFactory = $productCollectionFactory;
        $this->categoryCollectionFactory = $categoryCollectionFactory;
        $this->queueCollectionFactory = $queueCollectionFactory;
        $this->historicalCollectionFactory = $historicalCollectionFactory;
        $this->historicalFactory = $historicalFactory;
        $this->orderDataHelper = $orderDataHelper;
        $this->customerDataHelper = $customerDataHelper;
        $this->cartDataHelper = $cartDataHelper;
        $this->productDataHelper = $productDataHelper;
        $this->categoryDataHelper = $categoryDataHelper;
        $this->emulation = $emulation;
        $this->storeManager = $storeManager;
        $this->customerCollectionFactory = $customerCollectionFactory;
        $this->orderCollectionFactory = $orderCollectionFactory;
    }

    /**
     * Process historical data immediately with filters
     *
     * @param string $type
     * @param string|null $startDatetime
     * @param string|null $endDatetime
     * @param string|null $dateField
     * @param array|null $identifiers
     * @return array
     */
    public function processImmediateHistorical(
        $type,
        $startDatetime = null,
        $endDatetime = null,
        $dateField = null,
        $identifiers = null,
        $jobId = null
    ) {
        try {
            $page = 1;
            $processedItems = 0;

            do {
                $items = $this->getFilteredData(
                    $type,
                    $page,
                    $startDatetime,
                    $endDatetime,
                    $dateField,
                    $identifiers
                );

                if ($items && count($items) > 0) {
                    $this->createQueueEntries(
                        $items,
                        $this->getHelperByType($type),
                        $type,
                        $this->getPriorityByType($type),
                        $jobId
                    );
                    $processedItems += count($items);
                    $page++;
                } else {
                    break;
                }
            } while (count($items) == Settings::HISTORICAL_PAGE_SIZE);

            return ['message' => 'Processed ' . $processedItems . ' ' . $type . '(s)'];
        } catch (\Exception $e) {
            $this->_logger->critical($e);
            return ['error' => $e->getMessage()];
        }
    }

    /**
     * Get filtered data based on type and filters
     *
     * @param string $type
     * @param int $page
     * @param string|null $startDatetime
     * @param string|null $endDatetime
     * @param string|null $dateField
     * @param array|null $identifiers
     * @return iterable
     * @throws \InvalidArgumentException
     */
    protected function getFilteredData(
        $type,
        $page,
        $startDatetime,
        $endDatetime,
        $dateField,
        $identifiers
    ) {
        switch ($type) {
            case 'customer':
                return $this->getCustomerCollection(
                    $page,
                    $startDatetime,
                    $endDatetime,
                    $dateField,
                    $identifiers
                );
            case 'order':
                return $this->getOrderCollection(
                    $page,
                    $startDatetime,
                    $endDatetime,
                    $dateField,
                    $identifiers
                );
            case 'product':
                return $this->getProductCollection(
                    $page,
                    $startDatetime,
                    $endDatetime,
                    $dateField,
                    $identifiers
                );
            case 'category':
                return $this->getCategoryCollection(
                    $page,
                    $startDatetime,
                    $endDatetime,
                    $dateField,
                    $identifiers
                );
            default:
                throw new \InvalidArgumentException('Unknown historical type: ' . $type);
        }
    }

    /**
     * Get Customer Collection with filters
     *
     * @param int $page
     * @param string|null $startDatetime
     * @param string|null $endDatetime
     * @param string|null $dateField
     * @param array|null $identifiers
     * @return \Magento\Customer\Model\ResourceModel\Customer\Collection
     */
    protected function getCustomerCollection(
        $page,
        $startDatetime,
        $endDatetime,
        $dateField,
        $identifiers
    ) {
        /** @var \Magento\Customer\Model\ResourceModel\Customer\Collection $collection */
        $collection = $this->customerCollectionFactory->create();
        $collection->setPageSize(Settings::HISTORICAL_PAGE_SIZE)
            ->setCurPage($page);

        if ($identifiers) {
            if (is_array($identifiers)) {
                $collection->addFieldToFilter('email', ['in' => $identifiers]);
            } else {
                $collection->addFieldToFilter('email', ['eq' => $identifiers]);
            }
        }

        if ($startDatetime || $endDatetime) {
            $this->applyDateFilters($collection, $startDatetime, $endDatetime, $dateField);
        }

        return $collection;
    }

    /**
     * Get Order Collection with filters
     *
     * @param int $page
     * @param string|null $startDatetime
     * @param string|null $endDatetime
     * @param string|null $dateField
     * @param array|null $identifiers
     * @return \Magento\Sales\Model\ResourceModel\Order\Collection
     */
    protected function getOrderCollection(
        $page,
        $startDatetime,
        $endDatetime,
        $dateField,
        $identifiers
    ) {
        /** @var \Magento\Sales\Model\ResourceModel\Order\Collection $collection */
        $collection = $this->orderCollectionFactory->create();
        $collection->setPageSize(Settings::HISTORICAL_PAGE_SIZE)
            ->setCurPage($page);

        if ($identifiers) {
            if (is_array($identifiers)) {
                $collection->addFieldToFilter('entity_id', ['in' => $identifiers]);
            } else {
                $collection->addFieldToFilter('entity_id', ['eq' => $identifiers]);
            }
        }

        if ($startDatetime || $endDatetime) {
            $this->applyDateFilters($collection, $startDatetime, $endDatetime, $dateField);
        }

        return $collection;
    }

    /**
     * Get Product Collection with filters
     *
     * @param int $page
     * @param string|null $startDatetime
     * @param string|null $endDatetime
     * @param string|null $dateField
     * @param array|null $identifiers
     * @return ProductCollection
     */
    protected function getProductCollection(
        $page,
        $startDatetime,
        $endDatetime,
        $dateField,
        $identifiers
    ) {
        /** @var ProductCollection $collection */
        $collection = $this->productCollectionFactory->create();
        $collection->addAttributeToSelect('*')
            ->setPageSize(Settings::HISTORICAL_PAGE_SIZE)
            ->setCurPage($page);

        if ($identifiers) {
            if (is_array($identifiers)) {
                $collection->addFieldToFilter('entity_id', ['in' => $identifiers]);
            } else {
                $collection->addFieldToFilter('entity_id', ['eq' => $identifiers]);
            }
        }

        if ($startDatetime || $endDatetime) {
            $this->applyDateFilters($collection, $startDatetime, $endDatetime, $dateField);
        }

        return $collection;
    }

    /**
     * Get Category Collection with filters
     *
     * @param int $page
     * @param string|null $startDatetime
     * @param string|null $endDatetime
     * @param string|null $dateField
     * @param array|null $identifiers
     * @return CategoryCollection
     */
    protected function getCategoryCollection(
        $page,
        $startDatetime,
        $endDatetime,
        $dateField,
        $identifiers
    ) {
        /** @var CategoryCollection $collection */
        $collection = $this->categoryCollectionFactory->create();
        $collection->addAttributeToSelect('*')
            ->setPageSize(Settings::HISTORICAL_PAGE_SIZE)
            ->setCurPage($page);

        if ($identifiers) {
            if (is_array($identifiers)) {
                $collection->addIdFilter($identifiers);
            } else {
                $collection->addIdFilter([$identifiers]);
            }
        }

        if ($startDatetime || $endDatetime) {
            $this->applyDateFilters($collection, $startDatetime, $endDatetime, $dateField);
        }

        return $collection;
    }

    /**
     * Apply date filters to the collection
     *
     * @param $collection
     * @param string|null $startDatetime
     * @param string|null $endDatetime
     * @param string|null $dateField
     */
    protected function applyDateFilters($collection, $startDatetime, $endDatetime, $dateField)
    {
        if (!$startDatetime && !$endDatetime) {
            return;
        }

        $isEavCollection = $collection instanceof \Magento\Eav\Model\Entity\Collection\AbstractCollection;

        $fields = [];
        $conditions = [];

        $dateFields = [];
        if ($dateField === 'created' || $dateField === null || $dateField === 'both') {
            $dateFields[] = 'created_at';
        }
        if ($dateField === 'updated' || $dateField === null || $dateField === 'both') {
            $dateFields[] = 'updated_at';
        }

        foreach ($dateFields as $field) {
            $dateCondition = $this->buildDateCondition($startDatetime, $endDatetime);
            if (!empty($dateCondition)) {
                if ($isEavCollection) {
                    $conditionArray = array_merge(['attribute' => $field], $dateCondition);
                    $conditions[] = $conditionArray;
                } else {
                    $fields[] = $field;
                    $conditions[] = $dateCondition;
                }
            }
        }

        if (!empty($conditions)) {
            if ($isEavCollection) {
                $collection->addAttributeToFilter($conditions);
            } else {
                $collection->addFieldToFilter($fields, $conditions);
            }
        }
    }


    /**
     * Helper method to build date condition array
     *
     * @param string|null $startDatetime
     * @param string|null $endDatetime
     * @return array
     */
    protected function buildDateCondition($startDatetime, $endDatetime)
    {
        if ($startDatetime && $endDatetime) {
            return ['from' => $startDatetime, 'to' => $endDatetime];
        } elseif ($startDatetime) {
            return ['gteq' => $startDatetime];
        } elseif ($endDatetime) {
            return ['lteq' => $endDatetime];
        }
        return [];
    }


    /**
     * Get helper instance by type
     *
     * @param string $type
     * @return mixed
     * @throws \InvalidArgumentException
     */
    protected function getHelperByType($type)
    {
        switch ($type) {
            case 'customer':
                return $this->customerDataHelper;
            case 'order':
                return $this->orderDataHelper;
            case 'product':
                return $this->productDataHelper;
            case 'category':
                return $this->categoryDataHelper;
            default:
                throw new \InvalidArgumentException('Unknown helper type: ' . $type);
        }
    }

    /**
     * Get priority by type
     *
     * @param string $type
     * @return int
     */
    protected function getPriorityByType($type)
    {
        switch ($type) {
            case 'customer':
                return Settings::PRIORITY_CUSTOMER;
            case 'order':
                return Settings::PRIORITY_ORDER;
            case 'product':
                return Settings::PRIORITY_PRODUCT;
            case 'category':
                return Settings::PRIORITY_CATEGORY;
            default:
                return 5;
        }
    }

    /**
     * Trigger the historical process with a given job ID.
     *
     * @param string|null $jobId (optional) Job ID. If null, a default job ID will be generated.
     * @return array
     * @throws \Exception
     */
    public function triggerHistorical($jobId = null)
    {
        if ($jobId === null) {
            $jobId = '';
        }

        $this->resetAllHistoricalData();

        $this->triggerHistoricalForType('customer', $jobId);
        $this->triggerHistoricalForType('order', $jobId);
        $this->triggerHistoricalForType('category', $jobId);
        $this->triggerHistoricalForType('product', $jobId);

        $message = 'Historical data has been triggered';
        if ($jobId) {
            $message .= ' with Job ID: ' . $jobId;
        }

        return ['message' => $message, 'job_id' => $jobId];
    }


    /**
     * Trigger historical for a specific type
     *
     * @param $type
     * @param string|null $jobId
     * @param bool $removeOld
     *
     * @return array
     * @throws \Exception
     */
    public function triggerHistoricalForType($type, $jobId = null, $removeOld = false)
    {
        if ($jobId !== null) {
            $existingJob = $this->historicalCollectionFactory->create()
                ->addFieldToFilter('job_id', ['eq' => $jobId])
                ->addFieldToFilter('historical_type', ['eq' => $type])
                ->getFirstItem();

            if ($existingJob->getId()) {
                throw new \Magento\Framework\Exception\LocalizedException(__('Duplicate Job ID. A historical process with this ID already exists.'));
            }
        } else {
            $existingJob = $this->historicalCollectionFactory->create()
                ->addFieldToFilter('job_id', ['null' => true])
                ->addFieldToFilter('historical_type', ['eq' => $type])
                ->getFirstItem();

            if ($existingJob->getId()) {
                throw new \Magento\Framework\Exception\LocalizedException(__('A historical process with null Job ID already exists for this type.'));
            }
        }

        if ($removeOld) {
            $this->removeHistoricalQueueType($type, $jobId);
        }

        try {
            /** @var \Unific\Connector\Model\Historical $historical */
            $historical = $this->historicalFactory->create();
            $historical->setHistoricalType($type);
            $historical->setHistoricalTypePage(1);
            $historical->setPaused(0);
            $historical->setJobId($jobId);
            $historical->save();
        } catch (\Exception $e) {
            $this->_logger->critical($e);
        }

        $message = 'Historical data for type [' . $type . '] has been triggered';
        if ($jobId) {
            $message .= ' with Job ID ' . $jobId;
        }

        return ['message' => $message, 'job_id' => $jobId];
    }

    /**
     * Delete a type from the historical queue process
     *
     * @param string $type
     * @param string|null $jobId
     */
    public function removeHistoricalQueueType($type, $jobId = null)
    {
        /** @var HistoricalCollection $collection */
        $collection = $this->historicalCollectionFactory->create();
        $collection->addFieldToFilter('historical_type', ['eq' => $type]);

        if ($jobId === null) {
            // If jobId is null, only delete records with null jobId
            $collection->addFieldToFilter('job_id', ['null' => true]);
        } else {
            // Otherwise only delete records with specified jobId
            $collection->addFieldToFilter('job_id', ['eq' => $jobId]);
        }

        $collection->walk('delete');
    }

    /**
     * Inject historical data in the queue if historical is requested
     */
    public function queueHistorical()
    {
        /** @var HistoricalCollection $historicalCollection */
        $historicalCollection = $this->historicalCollectionFactory->create();
        $historicalCollection->addFieldToFilter('paused', ['eq' => 0]);

        if ($historicalCollection->getSize() === 0) {
            return;
        }

        foreach ($historicalCollection as $historicalItem) {
            try {
                $items = $this->processHistoricalItem($historicalItem);
                // Write back the latest ID to the historical queue manager
                if (count($items) == Settings::HISTORICAL_PAGE_SIZE) {
                    $this->incrementHistoricalQueuePage($historicalItem->getHistoricalType());
                } else {
                    $this->removeHistoricalQueueType($historicalItem->getHistoricalType());
                }
            } catch (\Throwable $e) {
                $this->_logger->critical($e);
                $this->incrementHistoricalQueuePage($historicalItem->getHistoricalType());
            }
        }
    }

    /**
     * Queue up all the historical data to be ready for sending
     */
    public function resetAllHistoricalData($jobId = null)
    {
        /** @var HistoricalCollection $collection */
        $collection = $this->historicalCollectionFactory->create();
        if ($jobId) {
            $collection->addFieldToFilter('job_id', ['eq' => $jobId]);
        }
        $collection->walk('delete');

        $this->resetHistoricalForType();

        return ['message' => 'Historical data has been stopped and reset'];
    }

    /**
     * Queue up all the historical data to be ready for sending
     * @param $type
     * @return array
     */
    public function resetHistoricalForType($type = null, $jobId = null)
    {
        if ($type) {
            $this->removeHistoricalQueueType($type, $jobId);
        }

        /** @var QueueCollection $queueCollection */
        $queueCollection = $this->queueCollectionFactory->create();
        $queueCollection->addFieldToFilter('headers', ['like' => '%' . $type . '/historical%']);
        if ($jobId) {
            $queueCollection->addFieldToFilter('job_id', ['eq' => $jobId]);
        }

        $queueCollection->walk('delete');

        return ['message' => 'Historical data for type [' . $type . '] has been stopped and reset'];
    }

    /**
     * Update the latest queued ID
     *
     * @param $type
     */
    protected function incrementHistoricalQueuePage($type)
    {
        /** @var HistoricalCollection $collection */
        $collection = $this->historicalCollectionFactory->create();
        $collection->addFieldToFilter('historical_type', ['eq', $type]);

        foreach ($collection as $item) {
            $item->setHistoricalTypePage($item->getHistoricalTypePage() + 1);
        }

        $collection->save();
    }

    /**
     * @param $subject
     * @return array
     */
    protected function getHeaders($subject)
    {
        $headers = [];
        $headers['X-SUBJECT'] = $subject;

        return $headers;
    }

    /**
     * @param \Unific\Connector\Model\Historical $historicalItem
     * @return mixed
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    protected function processHistoricalItem(\Unific\Connector\Model\Historical $historicalItem)
    {
        $items = $this->getTypeData($historicalItem, 'collection');
        [$type] = explode(
            false !== strpos((string)$historicalItem->getHistoricalType(), ':') ? ':' : '_',
            (string)$historicalItem->getHistoricalType()
        );

        if ($items) {
            $this->createQueueEntries(
                $items,
                $this->getTypeData($historicalItem, 'helper'),
                $type,
                $this->getTypeData($historicalItem, 'priority'),
                $historicalItem->getJobId()
            );
        }

        return $items;
    }

    /**
     * @param iterable $items
     * @param $helper
     * @param string $type
     * @param int $priority
     */
    protected function createQueueEntries($items, $helper, $type, $priority, $jobId = null)
    {
        $setEntityMethod = 'set' . ucwords($type);
        $getInfoMethod = 'get' . ucwords($type) . 'Info';

        foreach ($items as $item) {
            try {
                $helper->$setEntityMethod($item);
                if ($type === 'order') {
                    if ($item->hasShipments()) {
                        $helper->setShipment($item->getShipmentsCollection()->getLastItem());
                    }
                }
                if ((int)$item->getStoreId() > 0) {
                    try {
                        $this->emulation->startEnvironmentEmulation($item->getStoreId(), 'frontend');
                    } catch (\Throwable $e) {
                        $this->_logger->notice('Unable to emulate store ' . $item->getStoreId());
                    }
                }
                $headers = $this->getHeaders($type . '/historical');
                if ($jobId) {
                    $headers['X-UNIFIC-HISTORICAL-JOB-ID'] = $jobId;
                }

                $this->queueHelper->queue(
                    $this->scopeConfig->getValue('unific/webhook/' . $type . '_endpoint'),
                    $helper->$getInfoMethod(),
                    $priority,
                    $headers,
                    Request::METHOD_POST,
                    true,
                    null,
                    null,
                    Settings::QUEUE_HISTORICAL_MAX_RETRIES
                );

                $this->emulation->stopEnvironmentEmulation();
            } catch (\Throwable $e) {
                $this->_logger->critical($e);
                $this->emulation->stopEnvironmentEmulation();
            }
        }
    }


    /**
     * @param \Unific\Connector\Model\Historical $historical
     * @param string $element
     * @return mixed
     * @throws \Magento\Framework\Exception\LocalizedException
     * @todo if historical is set to save to file maybe increase Settings::HISTORICAL_PAGE_SIZE to make the job require less runs
     */
    protected function getTypeData(\Unific\Connector\Model\Historical $historical, string $element)
    {
        $helper = $element === 'helper';
        $type = (string)$historical->getHistoricalType();
        if (false !== strpos($historical->getHistoricalType(), ':')) {
            $elements = explode(':', (string)$historical->getHistoricalType());
            $type = $elements[0];
            unset($elements[0]);
            $store = count($elements) > 1 ? implode(':', $elements) : reset($elements);
        } elseif (false !== strpos($historical->getHistoricalType(), '_')) {
            [$type, $id] = array_pad(explode('_', (string)$historical->getHistoricalType()), 2, null);
        }
        switch ($type) {
            case 'customer':
                if ($element === 'collection') {
                    $this->searchCriteriaBuilder->setPageSize(Settings::HISTORICAL_PAGE_SIZE)
                        ->setCurrentPage($historical->getHistoricalTypePage());
                    if (isset($store)) {
                        $this->searchCriteriaBuilder->addFilter('store_id', ['eq' => $this->getStoreId($store)]);
                    }
                    if (isset($id)) {
                        $this->searchCriteriaBuilder->addFilter('entity_id', ['eq' => $id]);
                    }
                    $searchCriteria = $this->searchCriteriaBuilder->create();

                    return $this->customerRepository->getList($searchCriteria)->getItems();
                }
                return $helper ? $this->customerDataHelper : Settings::PRIORITY_CUSTOMER;
            case 'order':
                if ($element === 'collection') {
                    $this->searchCriteriaBuilder->setPageSize(Settings::HISTORICAL_PAGE_SIZE)
                        ->setCurrentPage($historical->getHistoricalTypePage());
                    if (isset($store)) {
                        $this->searchCriteriaBuilder->addFilter('store_id', ['eq' => $this->getStoreId($store)]);
                    }
                    if (isset($id)) {
                        $this->searchCriteriaBuilder->addFilter('entity_id', ['eq' => $id]);
                    }
                    $searchCriteria = $this->searchCriteriaBuilder->create();

                    return $this->orderRepository->getList($searchCriteria)->getItems();
                }
                return $helper ? $this->orderDataHelper : Settings::PRIORITY_ORDER;
            case 'product':
                if ($element === 'collection') {
                    /** @var ProductCollection $items */
                    $items = $this->productCollectionFactory->create();
                    $items->addAttributeToSelect('*')
                        ->setPageSize(Settings::HISTORICAL_PAGE_SIZE)
                        ->setCurPage($historical->getHistoricalTypePage());
                    if (isset($store) && $storeId = $this->getStoreId($store)) {
                        $items->addStoreFilter($storeId);
                    }
                    if (isset($id)) {
                        $items->addIdFilter($id);
                    }
                    $items->load()
                        ->addCategoryIds();

                    return $items;
                }
                return $helper ? $this->productDataHelper : Settings::PRIORITY_PRODUCT;
            case 'category':
                if ($element === 'collection') {
                    /** @var CategoryCollection $items */
                    $items = $this->categoryCollectionFactory->create();
                    $items->addAttributeToSelect('*')
                        ->setPageSize(Settings::HISTORICAL_PAGE_SIZE)
                        ->setCurPage($historical->getHistoricalTypePage());
                    if (isset($store)) {
                        $items->setStoreId($this->getStoreId($store));
                    }
                    if (isset($id)) {
                        $items->addIdFilter($id);
                    }

                    return $items;
                }
                return $helper ? $this->categoryDataHelper : Settings::PRIORITY_CATEGORY;
        }

        throw new \InvalidArgumentException('Unknown historical type: ' . $historical->getHistoricalType());
    }

    protected function getStoreId($storeCode): ?int
    {
        try {
            $store = $this->storeManager->getStore($storeCode);
            return $store->getId();
        } catch (NoSuchEntityException $e) {
            $this->_logger->warning('Cannot find store with code ' . $storeCode);
        }

        return null;
    }

    public function pauseHistorical($jobId)
    {
        /** @var HistoricalCollection $collection */
        $collection = $this->historicalCollectionFactory->create();
        if ($jobId) {
            $collection->addFieldToFilter('job_id', ['eq' => $jobId]);
        }
        foreach ($collection as $historicalItem) {
            $historicalItem->setPaused(1);
            $historicalItem->save();
        }

        return ['message' => 'All historical data processing has been paused'];
    }

    public function pauseHistoricalForType($type, $jobId)
    {
        /** @var HistoricalCollection $collection */
        $collection = $this->historicalCollectionFactory->create();
        $collection->addFieldToFilter('historical_type', ['eq' => $type]);
        if ($jobId) {
            $collection->addFieldToFilter('job_id', ['eq' => $jobId]);
        }
        foreach ($collection as $historicalItem) {
            $historicalItem->setPaused(1);
            $historicalItem->save();
        }

        return ['message' => 'Historical data processing for type [' . $type . '] has been paused'];
    }

    public function resumeHistorical($jobId = null)
    {
        /** @var HistoricalCollection $collection */
        $collection = $this->historicalCollectionFactory->create();
        $collection->addFieldToFilter('paused', ['eq' => 1]);
        if ($jobId) {
            $collection->addFieldToFilter('job_id', ['eq' => $jobId]);
        }
        foreach ($collection as $historicalItem) {
            $historicalItem->setPaused(0);
            $historicalItem->save();
        }

        return ['message' => 'All historical data processing has been resumed'];
    }

    public function resumeHistoricalForType($type, $jobId = null)
    {
        /** @var HistoricalCollection $collection */
        $collection = $this->historicalCollectionFactory->create();
        $collection->addFieldToFilter('historical_type', ['eq' => $type]);
        $collection->addFieldToFilter('paused', ['eq' => 1]);
        if ($jobId) {
            $collection->addFieldToFilter('job_id', ['eq' => $jobId]);
        }
        foreach ($collection as $historicalItem) {
            $historicalItem->setPaused(0);
            $historicalItem->save();
        }

        return ['message' => 'Historical data processing for type [' . $type . '] has been resumed'];
    }

}
