<?php

namespace Unific\Connector\Observer;

use Magento\Customer\Api\Data\CustomerInterface;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;
use Magento\Framework\Exception\LocalizedException;
use Unific\Connector\Helper\Data\Customer as CustomerDataHelper;
use Unific\Connector\Helper\Hmac;
use Unific\Connector\Helper\Message\Queue;
use Unific\Connector\Helper\Settings;

class CustomerLoginObserver implements ObserverInterface
{
    /**
     * @var ScopeConfigInterface
     */
    protected $scopeConfig;

    /**
     * @var CustomerDataHelper
     */
    protected $customerDataHelper;

    /**
     * @var Queue
     */
    protected $queueHelper;

    /**
     * @var Hmac
     */
    protected $hmacHelper;

    /**
     * @param ScopeConfigInterface $scopeConfig
     * @param CustomerDataHelper $customerDataHelper
     * @param Queue $queueHelper
     * @param Hmac $hmacHelper
     */
    public function __construct(
        ScopeConfigInterface $scopeConfig,
        CustomerDataHelper $customerDataHelper,
        Queue $queueHelper,
        Hmac $hmacHelper
    ) {
        $this->scopeConfig = $scopeConfig;
        $this->customerDataHelper = $customerDataHelper;
        $this->queueHelper = $queueHelper;
        $this->hmacHelper = $hmacHelper;
    }

    /**
     * Send customer/update webhook when customer logs in
     *
     * @param Observer $observer
     * @return void
     * @throws LocalizedException
     */
    public function execute(Observer $observer)
    {
        if ($this->scopeConfig->getValue('unific/connector/enabled') != 1) {
            return;
        }

        /** @var \Magento\Customer\Model\Customer $customer */
        $customer = $observer->getEvent()->getCustomer();

        if ($customer && $customer->getId()) {
            // Convert Customer Model to CustomerInterface
            $customerData = $customer->getDataModel();
            $this->customerDataHelper->setCustomer($customerData);

            $webhookEndpoint = $this->scopeConfig->getValue('unific/webhook/customer_endpoint');

            if ($webhookEndpoint && filter_var($webhookEndpoint, FILTER_VALIDATE_URL) !== false) {
                $headers = [];
                $headers['X-SUBJECT'] = 'customer/update';

                $this->queueHelper->queue(
                    $webhookEndpoint,
                    $this->customerDataHelper->getCustomerInfo(),
                    Settings::PRIORITY_CUSTOMER,
                    $headers,
                    \Laminas\Http\Request::METHOD_POST,
                    false,
                    null,
                    null,
                    Settings::QUEUE_MAX_RETRIES
                );
            }
        }
    }
}
